#!/bin/sh

cd $(dirname $0)
ROOTDIR=$(pwd)

#
# PLEASE DO NOT MODIFY THIS FILE IN ANY WAY
# Using this script after doing so may cause damage to the device
#

export PATH=$PATH:/bin:/sbin:/usr/bin/
export LD_LIBRARY_PATH=$LD_LIBRARY_PATH:/lib:/usr/lib

die()
{
	MESSAGE=""
	[ "$1" ] && MESSAGE="$1"
	echo "Fatal error $MESSAGE !" ; sync
	cp /mnt/src/lastupdate.log /mnt/src/last_failed_update_`date -u +"%d%m%Y_%H%M%S"`.log
	sync

	sleep 1

	psplash-write  "MSG ERROR UPDATING!!!
$MESSAGE

"
	psplash-write "PROGRESS 100"

	echo 0 > /sys/class/leds/us01:dl:usr0/brightness

	while [ "1" ]; do
		dbus-send --system --print-reply --dest=com.exor.EPAD "/Buzzer" com.exor.EPAD.Buzzer.beep int32:440 int32:100 &>/dev/null; sleep 1;
		echo 0 > /sys/class/leds/us01:fault:usr0/brightness; sleep 1;
		echo 1 > /sys/class/leds/us01:fault:usr0/brightness; sleep 1;
	done

	# should never reach this point
	exit -1
}

testPartition()
{
	mkdir -p /mnt/tmp || die "Error creating temp dir"
	mount $1 /mnt/tmp
	umount $1 || die "Failed umount test of $1"
	rm -rf /mnt/tmp
	echo "Mount test passed for partition $1"
}

md5check()
{
	[ -e $1.md5 ] || die "MD5 `basename $1`.md5 not found"
	# the md5 file format is md5<space><space>file , 2 spaces!!
	[ -n "`cat $1.md5 | cut -d ' ' -f 1`" ] || die "`basename $1` MD5 mismatch"
	md5sum $1 | grep -i "`cat $1.md5 | cut -d ' ' -f 1`" || die "`basename $1` MD5 mismatch"
}

untar()
{
	if [[ -e $1.tar ]]; then
 		tar xpf $1.tar || die "Errors untarring `basename $1.tar`"
	elif [[ -e $1.tar.gz ]]; then
		tar xzpf $1.tar.gz || die "Errors untarring `basename $1.tar.gz`"
	else
		die "File $1.tar.gz not found!"
	fi
}

untar_nop()
{
	if [[ -e $1.tar ]]; then
 		tar --no-same-permissions --no-same-owner -xf $1.tar || die "Errors untarring `basename $1.tar`"
	elif [[ -e $1.tar.gz ]]; then
		tar --no-same-permissions --no-same-owner -xzf $1.tar.gz || die "Errors untarring `basename $1.tar.gz`"
	else
		die "File $1.tar.gz not found!"
	fi
}

rmall()
{
	echo "FS cleanup..."
	rm -rf * .* etc/* ; sync
	echo
	echo "========================================"
	echo "Free space after cleanup : `df $(pwd)`" ;
	echo "----------------------------------------"
	find . ;
	echo "========================================"
	echo
	sync
}


formatPartition()
{
	echo "Format $1 as $2" ; sync
	umount $1 2> /dev/null
	mke2fs -m 1 -T ext4 -L $2 $1 || die "Error formatting $2"
	sync
}

backupPartition()
{
	TGZNAME="$1"
	CNAME="$2"
	PART="$3"
	MOUNTPOINT="$4"
	PROGRESS="$5"
	OS="$6"

	if [ -n "$backup" ]; then
		[ -e $PART ] || die "Partition $PART not found"
		testPartition $PART

		psplash-write "PROGRESS $PROGRESS"

		echo "Creating $CNAME backup..." ; sync
		psplashMSG "Creating $CNAME backup...\n
Writing...
"
		mkdir -p $BACKUP_DIR

		pushd .
		mkdir -p $MOUNTPOINT
		mkdir -p $BACKUP_DIR/$OS
		mount $PART $MOUNTPOINT || die "Failed mounting $PART"
		cd $MOUNTPOINT || die "Failed entering $CNAME folder"
		if ( ! tar cz --one-file-system -f $BACKUP_DIR/$OS/$TGZNAME.tar.gz . || ! ( md5sum  $BACKUP_DIR/$OS/$TGZNAME.tar.gz > $BACKUP_DIR/$OS/$TGZNAME.tar.gz.md5 )); then
			rm -rf $BACKUP_DIR/$OS/$TGZNAME.tar.gz*
			die "Failed creating $CNAME backup"
		fi

		psplashMSG "Creating $CNAME backup\n
Syncing...
"
		popd
		umount $PART
		sync
		echo "$CNAME backup creation successfull!"
		sync
	fi
}

updatePartition()
{
	TGZNAME="$1"
	CNAME="$2"
	PART="$3"
	PARTLABEL="$4"
	MOUNTPOINT="$5"
	PROGRESS="$6"

	if [ -e $ROOTDIR/$TGZNAME.tar.gz -o -e $ROOTDIR/$TGZNAME.tar ]; then
		[ -e $PART ] || die "Partition $PART not found"
		testPartition $PART

		psplash-write "PROGRESS $PROGRESS"

		echo "Updating $CNAME..." ; sync

		psplashMSG "Updating $CNAME\n
Checking MD5...
"
		if [ -e $ROOTDIR/$TGZNAME.tar.gz ] ; then
			md5check $ROOTDIR/$TGZNAME.tar.gz
		else
			md5check $ROOTDIR/$TGZNAME.tar
		fi

		psplashMSG "Updating $CNAME\n
Formatting $CNAME...
"
		formatPartition $PART $PARTLABEL

		psplashMSG "Updating $CNAME\n
Writing new $CNAME...
"
		pushd .
		mkdir -p $MOUNTPOINT
		mount $PART $MOUNTPOINT || die "Failed mounting $PART"
		cd $MOUNTPOINT || die "Failed entering $CNAME folder"
		untar $ROOTDIR/$TGZNAME
		psplashMSG "Updating $CNAME\n
Syncing...
"
		popd
		umount $PART
		sync
		echo "$CNAME update successfull!"
		sync

		return 0
	fi

		return 1
}

updateSplash() {
	PART=/dev/mmcblk1p1
	MOUNTPOINT=/mnt/factory
	if [ -e $ROOTDIR/splashimage.bin ]; then
		[ -e $PART ] || die "Partition $PART not found"
		testPartition $PART
		echo "Updating splashimage..." ; sync

		pushd .
		mkdir -p $MOUNTPOINT
		mount $PART $MOUNTPOINT || die "Failed mounting $PART"
		cd $MOUNTPOINT || die "Failed entering $CNAME folder"
		cp $ROOTDIR/splashimage.bin .

		popd
		umount $PART
		sync
		echo "splashimage update successfull!"
		sync
	fi
}

backupSplash() {
	PART=/dev/mmcblk1p1
	MOUNTPOINT=/mnt/factory
	PART_WCE7=/dev/mmcblk1p8
	MOUNTPOINT_WCE7=/mnt/wr

	if [ -n "$backup" ]; then
		[ -e $PART ] || die "Partition $PART not found"
		
		testPartition $PART
		echo "Creating splashimage backup..." ; sync

		mkdir -p $MOUNTPOINT
		mount $PART $MOUNTPOINT || die "Failed mounting $PART"

		if [ -e "$MOUNTPOINT/splashimage.bin" ]; then
			cp "$MOUNTPOINT/splashimage.bin" $BACKUP_DIR || die "Failed creating splashimage backup"
			sync
			echo "splashimage backup created successfully!"
		else
			echo "WARNING: splashimage file not found"
		fi

		umount $PART
		
		if [ -e  $ROOTDIR/wce7 ]; then
			[ -e $PART_WCE7 ] || die "Partition $PART_WCE7 not found"
			testPartition $PART_WCE7
			
			mkdir -p $MOUNTPOINT_WCE7
			mount $PART_WCE7 $MOUNTPOINT_WCE7 || die "Failed mounting $PART_WCE7"
			mkdir -p $BACKUP_DIR/wce7
			
			if [ -e "$MOUNTPOINT_WCE7/splashimage.bin" ]; then
				cp "$MOUNTPOINT_WCE7/splashimage.bin" $BACKUP_DIR/wce7 || die "Failed creating wce7 splashimage backup"
				sync
				echo "wce7 splashimage backup created successfully!"
			else
				echo "WARNING: wce7 splashimage file not found"
			fi

			umount $PART_WCE7
		
		fi
		
		sync
		echo "splashimage backup created successfully!"
		sync
	fi
}

backupWCE7Background() {

	PART=/dev/mmcblk1p8
	MOUNTPOINT=/mnt/wr

	if [ -n "$backup" ]; then
		[ -e $PART -a -e $ROOTDIR/wce7 ] || die "Partition $PART not found"
		
		testPartition $PART
		echo "Creating WCE7 background backup..." ; sync

		mkdir -p $MOUNTPOINT
		mount $PART $MOUNTPOINT || die "Failed mounting $PART"
		mkdir -p $BACKUP_DIR/wce7

		if [ -e "$MOUNTPOINT/background.bmp" ]; then
			cp "$MOUNTPOINT/background.bmp" $BACKUP_DIR/wce7 || die "Failed creating WCE7 background backup"
			sync
			echo "background backup created successfully!"
		else
			echo "WARNING: background  file not found"
		fi

		umount $PART
		
		sync
		echo "splashimage backup created successfully!"
		sync
	fi
}

clearCalibration()
{
	PART=/dev/mmcblk1p1
	MOUNTPOINT=/mnt/factory
	[ -e $PART ] || die "Partition $PART not found"

	mkdir -p $MOUNTPOINT
	mount $PART $MOUNTPOINT || die "Failed mounting $PART"
	rm -rf $MOUNTPOINT/etc/pointercal.xinput || die "Failed clearing calibration file"

	umount $PART
	sync
}

clearPartition()
{
	CNAME="$1"
	PART="$2"
	PARTLABEL="$3"
	MOUNTPOINT="$4"
	PROGRESS="$5"
	[ -e $PART ] || die "Partition $PART not found"
	testPartition $PART
	echo "Clearing $CNAME..." ; sync
	psplashMSG "Clearing $CNAME...\n

"
	psplash-write "PROGRESS $PROGRESS"

	formatPartition $PART $PARTLABEL
	pushd .
	mkdir -p $MOUNTPOINT
	mount $PART $MOUNTPOINT || die "Failed mounting $PART"
	cd $MOUNTPOINT || die "Failed entering $CNAME folder"
	psplashMSG "Clearing $CNAME\n
Syncing...
"
	popd
	umount $PART
	sync
	echo "$CNAME clearing successfull!"
	sync
}


psplashMSG () {
	psplash-write "MSG  ## $UPDATE_STEP ##
$( echo -e $1 )

"

}

echo
echo
echo
echo
echo "========================================"
echo "Update process starting at `date -u`"

[ "$1" = "clone" ] && clone_mode=1
[ -e $ROOTDIR/backup ] && backup=1

BACKUP_DIR="/mnt/src/backup_$( date -u +'%d%m%Y_%H%M%S' )"

# On both green and red LED
echo 1 > /sys/class/leds/us01:dl:usr0/brightness
echo 1 > /sys/class/leds/us01:fault:usr0/brightness

# start dbus
mkdir -p /var/run/dbus
/etc/init.d/dbus-1 start

# beep
dbus-send --system --print-reply --dest=com.exor.EPAD "/Buzzer" com.exor.EPAD.Buzzer.beep int32:880 int32:1000

rotation=0
[ -e /rotation ] && read rotation < /rotation

chvt 1
psplash --notouch --angle $rotation &

# starts a debug sshd session
(ifconfig eth0 10.10.10.207 up ; mount -t devpts devpts /dev/pts ; /etc/init.d/sshd start ) &

# sleep required to allow psplash starting and setting up display
sleep 1

psplash-write "MSG Starting system update 3 ..."
psplash-write "PROGRESS 1"
sleep 1
psplash-write  "MSG Starting system update 2 ..."
psplash-write "PROGRESS 2"
sleep 1
psplash-write  "MSG Starting system update 1 ..."
psplash-write "PROGRESS 3"
sleep 1

# Make sure the watchdog is not active
echo V > /dev/watchdog

# this is hard-code  mounted by psplash to read the splash image and can now be umounted
umount /dev/mmcblk1p1

echo "Update process starting at `date -u`" sync

echo "======================================="
echo "Current fs setup:"
echo "---------------------------------------"
cat /proc/mounts
echo "---------------------------------------"
df
echo "======================================="
echo
echo
echo
echo "======================================="
echo "Running processes:"
echo "---------------------------------------"
ps aux
echo "======================================="
sync

DRIVE=/dev/mmcblk1
[ -e $DRIVE ] || "Drive $DRIVE not found!"

# ====================================================================
#                         BACKUP PROCESS
# ====================================================================

UPDATE_STEP="BSP BACKUP"

if [ -n "$backup" ]; then

	mkdir -p "$BACKUP_DIR"

	cmps="$( cat /mnt/src/src/backup 2>/dev/null | sed 's:\(^[0-9a-z_]*\).*:\1:' )"

	# If the file is empty the default is to save everything
	if [ -z "$cmps" ]; then
		cmps="$( echo -e "mainos\nconfigos\ndata\netc\nfactory\nbootloader\nxloader\nsplash" )"
		[ -e  $ROOTDIR/wce7 ] && cmps="$cmps$( echo -e "\nwce7_system\nwce7_data\nwce7_registry\nwce7_package\nwce7_background")"
	fi
	
	echo -e "Starting BSP backup for following components:\n$cmps\n"

	if ( echo "$cmps" | grep -q "^bootloader *$" ); then

	        echo "Creating u-boot backup..." ; sync
        	psplashMSG  "Creating u-boot backup...\n
Writing...
"
		dd if=/dev/mmcblk1boot0 of=$BACKUP_DIR/u-boot.img bs=1024 count=384 || die "Failed creating u-boot backup"
		md5sum $BACKUP_DIR/u-boot.img > $BACKUP_DIR/u-boot.img.md5 || die "Failed creating u-boot backup"

        	echo "u-boot backup creation successful!";
		psplashMSG  "Creating u-boot backup...\n
Syncing...
"
	fi

	if ( echo "$cmps" | grep -q "^xloader *$" ); then

        	echo "Creating preloader backup..." ; sync
        	psplashMSG  "Creating preloader backup...\n
Writing...
"
		dd if=$DRIVE of=$BACKUP_DIR/MLO.raw bs=512 skip=256 count=255 || die "Failed creating preloader backup"
		md5sum $BACKUP_DIR/MLO.raw > $BACKUP_DIR/MLO.raw.md5 || die "Failed creating preloader backup"

        	echo "Preloader backup creation successful!";
		psplashMSG  "Creating preloader backup...\n
Syncing...
"
		sync
	fi
	
	if ( echo "$cmps" | grep -q "^wce7_system *$" ); then

		[ -e  /dev/mmcblk1p7 ] || die "Partition /dev/mmcblk1p7 not found"
		testPartition /dev/mmcblk1p7

		psplash-write "PROGRESS 10"

		echo "Creating WCE7 system backup..." ; sync
		psplashMSG "Creating WCE7 system backup...\n
Writing...
"
		mkdir -p $BACKUP_DIR/wce7
		mkdir -p /mnt/ws
		mount /dev/mmcblk1p7 /mnt/ws || die "Failed mounting /dev/mmcblk1p7"
		if ( ! ( cat /mnt/ws/version.bin  /mnt/ws/wceimage.bin > $BACKUP_DIR/wce7/system.bin ) || 
		     ! ( md5sum  $BACKUP_DIR/wce7/system.bin > $BACKUP_DIR/wce7/system.bin.md5 )); then
			rm -rf $BACKUP_DIR/wce7/system.bin*
			die "Failed creating  WCE7 system backup"
		fi

		psplashMSG "Creating  WCE7 system backup\n
Syncing...
"

		umount /dev/mmcblk1p7
		sync
		echo "WCE7 system backup creation successfull!"
		sync
	fi
	
	if ( echo "$cmps" | grep -q "^wce7_data *$" ); then

		[ -e /dev/mmcblk1p10 ] || die "Partition /dev/mmcblk1p10 not found"

		psplash-write "PROGRESS 20"

		echo "Creating WCE7 data backup..." ; sync
		psplashMSG "Creating WCE7 data backup...\n
Writing...
"
		mkdir -p $BACKUP_DIR/wce7
		
		partSize="$( fdisk -l /dev/mmcblk1p10 | grep Disk | awk '{print $5}' )"

		# The data image is a gzip of the raw partition plus 8byte where we store the uncompressed data size
		if ( ! ( cat /dev/mmcblk1p10 | gzip > $BACKUP_DIR/wce7/data.bin && echo -ne "$(printf '%016x' $partSize | sed 's:\(..\):\\x\1:g')" >> $BACKUP_DIR/wce7/data.bin ) || 
		     ! ( md5sum  $BACKUP_DIR/wce7/data.bin > $BACKUP_DIR/wce7/data.bin.md5 )); then
			rm -rf $BACKUP_DIR/wce7/data.bin*
			die "Failed creating  WCE7 system backup"
		fi

		psplashMSG "Creating  WCE7 system backup\n
Syncing...
"

		sync
		echo "WCE7 system backup creation successfull!"
		sync
	fi

	( echo "$cmps" | grep -q "^mainos *$" ) && backupPartition mainos mainos /dev/mmcblk1p3 /mnt/m 30
	( echo "$cmps" | grep -q "^configos *$" ) && backupPartition configos configos /dev/mmcblk1p2  /mnt/c 60
	( echo "$cmps" | grep -q "^etc *$" ) && backupPartition etc "etc data" /dev/mmcblk1p5 /mnt/e 82
	( echo "$cmps" | grep -q "^factory *$" ) && backupPartition factory "factory data" /dev/mmcblk1p1 /mnt/f 85
	( echo "$cmps" | grep -q "^data *$" ) && backupPartition data "user data" /dev/mmcblk1p6 /mnt/d 90
	( echo "$cmps" | grep -q "^wce7_package *$" ) && backupPartition package "WCE7 package" /dev/mmcblk1p9 /mnt/wp 93 wce7
	( echo "$cmps" | grep -q "^wce7_registry *$" ) && backupPartition registry "WCE7 registry" /dev/mmcblk1p8 /mnt/wr 98 wce7
	( echo "$cmps" | grep -q "^splash *$" ) && backupSplash
	( echo "$cmps" | grep -q "^wce7_background *$" ) && backupWCE7Background
	
fi


# ====================================================================
#                         UPDATE PROCESS
# ====================================================================

UPDATE_STEP="BSP UPDATE"

psplash-write "PROGRESS 0"

# SS 24.07.2015 copy u-boot only once at beginning to reduce chance of mmc hang for "old" kernels
if [ -e $ROOTDIR/u-boot.img ]; then
	echo "Updating u-boot..." ; sync
	psplashMSG  "Updating u-boot...\n
Checking md5...
"
	psplash-write "PROGRESS 1"

	md5check $ROOTDIR/u-boot.img

	psplashMSG  "Updating u-boot...\n
Remounting rw...
"

	echo "Disabling ro in u-boot..." ; sync
	echo 0 > /sys/block/mmcblk1boot0/force_ro || die "Enabling rw in uboot"

	psplashMSG "Updating u-boot...\n
Writing...
"
	dd if=$ROOTDIR/u-boot.img of=/dev/mmcblk1boot0 bs=512 seek=0 || die "Writing bootloader"
	echo "Enabling ro in u-boot..." ; sync

	psplashMSG "Updating u-boot...\n
Syncing...
"
	echo 1 > /sys/block/mmcblk1boot0/force_ro
	echo "uboot update successful!"
	psplashMSG "Updating u-boot...\n
Syncing...
"
sync
fi


if [ -e $ROOTDIR/reformat -o -n "$clone_mode" ] ; then
	psplashMSG "Reformatting whole machine...\n
Checking md5...
"
	echo "Checking reformat dependencies..."
	[ -e $ROOTDIR/u-boot.img ] || die "u-boot.img required after reformat"
	[ -e $ROOTDIR/MLO.raw ] || die "MLO.raw required after reformat"
	[ -e $ROOTDIR/configos.tar.gz -o -e $ROOTDIR/configos.tar ] || die "configos required after reformatting"
	md5check $ROOTDIR/u-boot.img
	md5check $ROOTDIR/MLO.raw
	if [ -e $ROOTDIR/configos.tar.gz ] ; then
		 md5check $ROOTDIR/configos.tar.gz
	else
		 md5check $ROOTDIR/configos.tar
	fi

	if [ -n "$clone_mode" ]; then

		[ -e $ROOTDIR/mainos.tar.gz -o -e $ROOTDIR/mainos.tar ] || die "mainos required in clone mode"
		if [ -e $ROOTDIR/mainos.tar.gz ] ; then
			md5check $ROOTDIR/mainos.tar.gz
		else
			md5check $ROOTDIR/mainos.tar
		fi

		[ -e $ROOTDIR/data.tar.gz -o -e $ROOTDIR/data.tar ] || die "data required in clone mode"
		if [ -e $ROOTDIR/data.tar.gz ] ; then
			md5check $ROOTDIR/data.tar.gz
		else
			md5check $ROOTDIR/data.tar
		fi

		[ -e $ROOTDIR/factory.tar.gz -o -e $ROOTDIR/factory.tar -o -e $ROOTDIR/splashimage.bin ] || die "factory required in clone mode"
		if [ -e $ROOTDIR/factory.tar.gz ] ; then
			md5check $ROOTDIR/factory.tar.gz
		elif [ -e $ROOTDIR/factory.tar ] ; then
			md5check $ROOTDIR/factory.tar
		fi

	fi

	# Erase any existent partition table and get device size
	psplashMSG "Reformatting whole machine...\n
Erasing partitions...
"

	cat /proc/mounts

	echo
	echo "Erasing any existing partition and get media size..."
	dd if=/dev/zero of=$DRIVE bs=1024 count=1024 || die "Error erasing partitions..."
	SIZE=`fdisk -l $DRIVE | grep Disk | awk '{print $5}'`
	echo DISK SIZE - $SIZE bytes
	CYLINDERS=$(( $(( $SIZE )) / 255 / 63 /512 ))
	echo CYLINDERS - $CYLINDERS

	# Disk partitioning
	echo
	echo "Media partitioning..."

	if [ "$SIZE" -gt "3000000000" ]; then
		psplashMSG  "Reformatting whole machine...\n
Partitioning 4GB...
"
		echo "4GB EMMC detected: creating partitions for Linux and Android..."
		{ 
		echo 2,8,L
		echo 10,44,L 
		echo 54,64,L 
		echo 118,,E
		echo 118,8,L
		echo 126,108,L
		echo 234,38,L
		echo 272,64,L
		echo 336,38,L
		echo 374,,L
		} | sfdisk --force -D -H 255 -S 63 -C $CYLINDERS $DRIVE || die "Error partitioning 4GB"
	else
		psplashMSG "Reformatting whole machine...\n
Partitioning 2GB...
"
		echo "2GB EMMC detected: creating partitions for Linux only..."
		{ 
		echo 2,8,L
		echo 10,44,L 
		echo 54,64,L 
		echo 118,,E
		echo 118,8,L
		echo 126,,L
		} | sfdisk --force -D -H 255 -S 63 -C $CYLINDERS $DRIVE || die "Error partitioning 2GB"
	fi

	psplashMSG "Reformatting whole machine...\n
Formatting...
"
	# Partitions formatting
	echo
	echo "Partitions formatting..."
	mke2fs -m 1 -T ext4 -L "factory" $DRIVE'p1'
	mke2fs -m 1 -T ext4 -L "configos" $DRIVE'p2'
	mke2fs -m 1 -T ext4 -L "mainos" $DRIVE'p3'
	mke2fs -m 1 -T ext4 -L "etc" $DRIVE'p5'
	mke2fs -m 1 -T ext4 -L "data" $DRIVE'p6'

	# Clear FRAM
	[ -e /dev/fram ] && dd if=/dev/zero of=/dev/fram bs=1024 count=64

fi

if [ -e $ROOTDIR/MLO.raw ]; then
        echo "Updating preloader..." ; sync
        psplashMSG  "Updating preloader...\n
Checking md5...
"
	md5check $ROOTDIR/MLO.raw

	psplashMSG "Updating preloader...\n
Writing...
"
	# Write the MLO.raw
	echo
	echo "Writing the preloader (MLO.raw)..."

	dd if=$ROOTDIR/MLO.raw of=$DRIVE bs=512 seek=256 || die "Error writing preloader (1)"
	dd if=$ROOTDIR/MLO.raw of=$DRIVE bs=512 seek=512 || die "Error writing preloader (2)"
	dd if=$ROOTDIR/MLO.raw of=$DRIVE bs=512 seek=768 || die "Error writing preloader (3)"
fi

sync

updatePartition mainos mainos /dev/mmcblk1p3 mainos /mnt/m 30
[ "$?" -eq "0" ] && clearPartition etc /dev/mmcblk1p5 etc /mnt/e 35

updatePartition configos configos /dev/mmcblk1p2 configos /mnt/c 60
updatePartition etc "etc data" /dev/mmcblk1p5 etc /mnt/e 82
updatePartition factory "factory data" /dev/mmcblk1p1 factory /mnt/f 85
[ "$?" -eq "0" ] && clearCalibration

updatePartition data "user data" /dev/mmcblk1p6 data /mnt/d 90

# If data partition changed, clear FRAM
if [ "$?" -eq "0" -a -e /dev/fram ]; then
	echo
    echo Clearing FRAM...
    dd if=/dev/zero of=/dev/fram bs=1024 count=64
fi

updateSplash

# If the system has an Android compatible partitioning and the src/android folder exists, try to install Android
if [ -e $ROOTDIR/android ]; then
if [ -e /dev/mmcblk1p10 ]; then
	echo "Updating Android..." ; sync
	psplashMSG "Updating Android...\n

"
	psplash-write "PROGRESS 92"
	# Unmount all Android related partitions
	umount /dev/mmcblk1p7 2>/dev/null
	umount /dev/mmcblk1p8 2>/dev/null
	umount /dev/mmcblk1p9 2>/dev/null
	umount /dev/mmcblk1p10 2>/dev/null

	#format and install the Android system
	if [ -e $ROOTDIR/android/system.tar.gz -o -e $ROOTDIR/android/system.tar ]; then
		psplashMSG "Updating Android system...\n
Checking md5...
"
		if [ -e $ROOTDIR/android/system.tar.gz ] ; then
			md5check $ROOTDIR/android/system.tar.gz
		else
			md5check $ROOTDIR/android/system.tar
		fi

		psplashMSG "Updating Android system...\n
Creating partitions...
"

		echo "Updating Android system..." ; sync
		mke2fs -m 1 -T ext4 -L "android_vendor" /dev/mmcblk1p7 || die "Formatting android_vendor"
		mke2fs -T ext4 -L "android_system" /dev/mmcblk1p8 || die "Formatting android_system"
		mke2fs -T ext4 -L "android_cache" /dev/mmcblk1p9 || die "Formatting android_cache"

		psplashMSG "Updating Android system...\n
Writing data...
"

		pushd .
		mkdir -p /mnt/android_system
		mount /dev/mmcblk1p8 /mnt/android_system || die "Mounting android_system"
		cd /mnt/android_system || die "Entering android_system folder"
		untar $ROOTDIR/android/system
		psplash-write "PROGRESS 93"
		popd

		psplashMSG "Updating Android system...\n
Syncing...
"
		umount /dev/mmcblk1p8
		echo "Android system successful!"
		sync
		psplash-write "PROGRESS 94"
	fi

	#update Android data, if any
	if [ -e $ROOTDIR/android/data.tar.gz -o -e $ROOTDIR/android/data.tar ]; then
		echo "Updating Android data..." ; sync
		psplashMSG  "Updating Android data...\n
Checking md5...
"
		if [ -e $ROOTDIR/android/data.tar.gz ] ; then
			md5check $ROOTDIR/android/data.tar.gz
		else
			md5check $ROOTDIR/android/data.tar
		fi

		psplashMSG "Updating Android data...\n
Creating data partitions...
"
		mke2fs -T ext4 -L "android_data" /dev/mmcblk1p10 || die "Formatting android_data"

		psplashMSG "Updating Android data...\n
Writing data...
"
		pushd .
		mkdir -p /mnt/android_data
		mount /dev/mmcblk1p10 /mnt/android_data || die "Mounting android_data"
		cd /mnt/android_data || die "Entering android_data folder"
		untar $ROOTDIR/android/data
		popd

		psplashMSG "Updating Android data...\n
Syncing...
"
		umount /dev/mmcblk1p10
		psplash-write "PROGRESS 95"
		echo "Android system successful!"
		sync
	fi

	#update Android vendor, if any
	if [ -e $ROOTDIR/android/vendor.tar.gz -o -e $ROOTDIR/android/vendor.tar ]; then
		echo "Updating Android vendor..." ; sync
		psplashMSG "Updating Android vendor...\n
Checking md5...
"
		if [ -e $ROOTDIR/android/vendor.tar.gz ] ; then
			md5check $ROOTDIR/android/vendor.tar.gz
		else
			md5check $ROOTDIR/android/vendor.tar
		fi

		psplashMSG "Updating Android vendor...\n
Writing data...
"
		pushd .
		mkdir -p /mnt/android_vendor
		mount /dev/mmcblk1p7 /mnt/android_vendor || die "Mounting android_vendor"
		cd /mnt/android_vendor || die "Entering android_vendor folder"
		untar $ROOTDIR/android/vendor
		popd

		psplashMSG "Updating Android vendor...\n
Syncing...
"
		umount /dev/mmcblk1p7
		psplash-write "PROGRESS 97"
		echo "Android system successful!"
		sync
	fi
fi
fi

# If the system has an WCE7 compatible partitioning and the src/wce7 folder exists, try to install WCE7
if [ ! -e $ROOTDIR/android -a -e $ROOTDIR/wce7 ]; then
if [ -e /dev/mmcblk1p10 ]; then
	echo "Updating WCE7..." ; sync
	psplashMSG "Updating WCE7...\n

"
	psplash-write "PROGRESS 92"
	# Unmount all WCE7 related partitions
	umount /dev/mmcblk1p7 2>/dev/null
	umount /dev/mmcblk1p8 2>/dev/null
	umount /dev/mmcblk1p9 2>/dev/null
	umount /dev/mmcblk1p10 2>/dev/null

	#format and install the WCE7 system
	if [ -e $ROOTDIR/wce7/system.bin ]; then
		psplashMSG "Updating WCE7 system...\n
Checking md5...
"
		if [ -e $ROOTDIR/wce7/system.bin ] ; then
			md5check $ROOTDIR/wce7/system.bin
		fi

		psplashMSG "Updating WCE7 system...\n
Creating partitions...
"

		echo "Updating WCE7 system..." ; sync
		( mkdosfs -F32 -f1 -s1 /dev/mmcblk1p7 && sfdisk --id /dev/mmcblk1 7 c ) || die "Formatting wce7_system"

		psplashMSG "Updating WCE7 system...\n
Writing data...
"

		mkdir -p /mnt/wce7_system
		mount /dev/mmcblk1p7 /mnt/wce7_system || die "Mounting wce7_system"
		dd if="$ROOTDIR/wce7/system.bin" bs=32 count=1 of="/mnt/wce7_system/version.bin" || die "Error writing WCE7 system (1)"
		tail -c +33 "$ROOTDIR/wce7/system.bin" > "/mnt/wce7_system/wceimage.bin"  || die "Error writing WCE7 system (2)"
		
		psplash-write "PROGRESS 93"

		psplashMSG "Updating WCE7 system...\n
Syncing...
"
		umount /dev/mmcblk1p7
		echo "WCE7 system successful!"
		sync
		psplash-write "PROGRESS 94"
	fi

	#update WCE7 registry, if any
	if [ -e $ROOTDIR/wce7/registry.tar.gz -o -e $ROOTDIR/wce7/registry.tar ]; then
		echo "Updating WCE7 registry..." ; sync
		psplashMSG "Updating WCE7 registry...\n
Checking md5...
"
		if [ -e $ROOTDIR/wce7/registry.tar.gz ] ; then
			md5check $ROOTDIR/wce7/registry.tar.gz
		else
			md5check $ROOTDIR/wce7/registry.tar
		fi

		psplashMSG "Updating WCE7 registry...\n
Writing data...
"
		( mkdosfs -F32 -f1 -s1 /dev/mmcblk1p8 && sfdisk --id /dev/mmcblk1 8 c ) || die "Formatting wce7_registry"

		pushd .
		mkdir -p /mnt/wce7_reg
		mount /dev/mmcblk1p8 /mnt/wce7_reg || die "Mounting wce7_registry"
		cd /mnt/wce7_reg || die "Entering wce7_registry folder"
		untar_nop $ROOTDIR/wce7/registry
		popd

		psplashMSG "Updating WCE7 registry...\n
Syncing...
"
		umount /dev/mmcblk1p8
		psplash-write "PROGRESS 97"
		echo "WCE7 registry successful!"
		sync
	fi

	#update WCE7 data, if any
	if [ -e $ROOTDIR/wce7/data.bin ]; then
		echo "Updating WCE7 data..." ; sync
		psplashMSG  "Updating WCE7 data...\n
Checking md5...
"
		if [ -e $ROOTDIR/wce7/data.bin ] ; then
			md5check $ROOTDIR/wce7/data.bin 
		fi

		psplashMSG "Updating WCE7 data...\n
Creating data partitions...
"
		sfdisk --id /dev/mmcblk1 10 7 || die "Formatting wce7_data"

		psplashMSG "Updating WCE7 data...\n
Writing data...
"

		# Last 8 bytes of the image are the size of the uncompressed data
		gzipSize="$(ls -l "$ROOTDIR/wce7/data.bin" | awk '{ print $5}')"
		dataSize="$(tail -c +$((${gzipSize}-7)) "$ROOTDIR/wce7/data.bin" | hexdump -e '16/1 "%02x"')"
		dataSize="$(( 0x${dataSize} ))"
		partSize="$( fdisk -l /dev/mmcblk1p10 | grep Disk | awk '{print $5}' )"

		echo partSize $partSize, dataSize $dataSize

		# Check data does not exceeds partition size
		[ "$dataSize" -gt "$partSize" ] &&  die "wce7_data exceeds partition size"

		# Read the gzip file until the last 8 bytes, the two dd commands
		# have the same effect as "head -c" which is not supported by busybox
		count="$(( ($gzipSize -8 ) /512 ))"
		( ( dd if="$ROOTDIR/wce7/data.bin" bs=512 count=$count && dd if="$ROOTDIR/wce7/data.bin" bs=1 skip=$((  512*$count )) count=$(( $gzipSize -  (512*$count) -8 )) ) | gunzip > /dev/mmcblk1p10 ) || die "Error writing WCE7 data"

		# Write an empty "calibration" file which will force a WCE7 recalibration.
		# This is done to avoid applying calibration parameters from the panel from which the data backup was taken
		mkdir -p /mnt/wce7_reg
		mount /dev/mmcblk1p8 /mnt/wce7_reg || die "Mounting wce7_registry"
		rm -rf /mnt/wce7_reg/calibration
		touch /mnt/wce7_reg/calibration
		umount /dev/mmcblk1p8

		psplashMSG "Updating WCE7 data...\n
Syncing...
"

		psplash-write "PROGRESS 95"
		echo "WCE7 data successful!"
		sync
	fi

	#update WCE7 package, if any
	if [ -e $ROOTDIR/wce7/package.tar.gz -o -e $ROOTDIR/wce7/package.tar ]; then
		echo "Updating WCE7 package..." ; sync
		psplashMSG "Updating WCE7 package...\n
Checking md5...
"
		if [ -e $ROOTDIR/wce7/package.tar.gz ] ; then
			md5check $ROOTDIR/wce7/package.tar.gz
		else
			md5check $ROOTDIR/wce7/package.tar
		fi

		psplashMSG "Updating WCE7 package...\n
Writing data...
"
		( mkdosfs -F32 -f1 -s1 /dev/mmcblk1p9 && sfdisk --id /dev/mmcblk1 9 c ) || die "Formatting wce7_package"

		pushd .
		mkdir -p /mnt/wce7_pkg
		mount /dev/mmcblk1p9 /mnt/wce7_pkg || die "Mounting wce7_package"
		cd /mnt/wce7_reg || die "Entering wce7_package folder"
		untar_nop $ROOTDIR/wce7/package
		popd

		psplashMSG "Updating WCE7 package...\n
Syncing...
"
		umount /dev/mmcblk1p9
		psplash-write "PROGRESS 97"
		echo "WCE7 package successful!"
		sync
	fi

	#update WCE7 background, if any
	if [ -e $ROOTDIR/wce7/background.bmp -o -e $ROOTDIR/splashimage.bin -o -e $ROOTDIR/wce7/splashimage.bin ]; then

		mkdir -p /mnt/wce7_reg
		mount /dev/mmcblk1p8 /mnt/wce7_reg || die "Mounting wce7_registry"

		if [ -e $ROOTDIR/wce7/background.bmp ]; then
			psplashMSG "Updating WCE7 background...\n
Writing data...
"
			cp $ROOTDIR/wce7/background.bmp /mnt/wce7_reg || die "Error writing WCE7 background"
			sync
		fi
		
		if [ -e $ROOTDIR/splashimage.bin -o $ROOTDIR/wce7/splashimage.bin ]; then
			psplashMSG "Updating WCE7 splashimage...\n
Writing data...
"
			if [ -e $ROOTDIR/wce7/splashimage.bin ]; then
				cp $ROOTDIR/wce7/splashimage.bin /mnt/wce7_reg || die "Error writing WCE7 splashimage"
			else
				cp $ROOTDIR/splashimage.bin /mnt/wce7_reg || die "Error writing WCE7 splashimage"
			fi
			sync
		fi
		
		umount /dev/mmcblk1p9
	fi

fi
fi

echo "Update complete, checking partitions"
psplashMSG "Update complete, checking partitions...\n
"

psplash-write "PROGRESS 99"

# Ignore fsck time related errors
echo '[options]' > /etc/e2fsck.conf
echo 'broken_system_clock = 1' >> /etc/e2fsck.conf

# fsck linux partitions
for i in 1 2 3 5 6 ; do
	[ -e /dev/mmcblk1p$i ] || continue ;
	umount /dev/mmcblk1p$i 2> /dev/null
	head /dev/mmcblk1p$i > /dev/null || continue ;
	fsck.ext4 -fp /dev/mmcblk1p$i
	if (($? > 3)); then
		if (($i < 8)); then
			die "Unrecoverable error in partition /dev/mmcblk1p$i"
		else
			echo "Unrecoverable error in partition /dev/mmcblk1p$i"	# extended partitions might be unreachable in old kernels
		fi
	fi
	sync
done

if [ -e $ROOTDIR/android ]; then

	# fsck android partitions
	for i in 7 8 9 10 ; do
		[ -e /dev/mmcblk1p$i ] || continue ;
		umount /dev/mmcblk1p$i 2> /dev/null
		head /dev/mmcblk1p$i > /dev/null || continue ;
		fsck.ext4 -fp /dev/mmcblk1p$i
		if (($? > 3)); then
			if (($i < 8)); then
				die "Unrecoverable error in partition /dev/mmcblk1p$i"
			else
				echo "Unrecoverable error in partition /dev/mmcblk1p$i" # extended partitions might be unreachable in old kernels
			fi
		fi
		sync
	done

fi

if [ -e $ROOTDIR/wce7 -a ! -e $ROOTDIR/android ]; then

	# fsck WCE7 FAT32 partitions
	for i in 7 8 9 ; do
		[ -e /dev/mmcblk1p$i ] || continue ;
		umount /dev/mmcblk1p$i 2> /dev/null
		head /dev/mmcblk1p$i > /dev/null || continue ;
		dosfsck -a -V /dev/mmcblk1p$i
		if (($? > 0)); then
			die "Unrecoverable error in partition /dev/mmcblk1p$i"
		fi
		sync
	done

fi

echo "Update complete, autorebooting..." ; sync
psplash-write  "MSG Process completed succesfully!
Unplug the USB stick and
power off the machine
"

psplash-write "PROGRESS 100"

# reboot countdown
dbus-send --system --print-reply --dest=com.exor.EPAD "/Buzzer" com.exor.EPAD.Buzzer.beep int32:440 int32:100; sleep 1;
dbus-send --system --print-reply --dest=com.exor.EPAD "/Buzzer" com.exor.EPAD.Buzzer.beep int32:440 int32:100; sleep 1;
dbus-send --system --print-reply --dest=com.exor.EPAD "/Buzzer" com.exor.EPAD.Buzzer.beep int32:440 int32:100; sleep 1;

/etc/init.d/dbus-1 stop
killall EPAD
killall psplash

sync

[ -n "$clone_mode" ] && reboot -f
